<?php

namespace App\Controllers\Admin;

use CodeIgniter\Controller;
use App\Models\UserModel;

class Dashboard extends Controller
{
    protected $userModel;

    public function __construct()
    {
        $this->userModel = new UserModel();
    }

    public function index()
    {
        // Verificar autenticación
        if (!session()->get('isLoggedIn')) {
            return redirect()->to(base_url('login'));
        }

        $role = session()->get('role');
        $userId = session()->get('user_id');
        $username = session()->get('username');

        // Obtener estadísticas según el rol
        if ($role === 'admin') {
            $stats = $this->getAdminStats();
            $notifications = $this->getAdminNotifications();
            $recentActivity = $this->getRecentActivity();
        } else {
            $stats = $this->getAdvisorStats($userId);
            $notifications = $this->getAdvisorNotifications($userId);
            $recentActivity = $this->getAdvisorActivity($userId);
        }

        $data = [
            'title' => 'Panel de Administración - Chat Solibot',
            'role' => $role,
            'username' => $username,
            'userId' => $userId,
            'stats' => $stats,
            'notifications' => $notifications,
            'recentActivity' => $recentActivity
        ];

        return view('admin/dashboard', $data);
    }

    /**
     * Estadísticas para administradores
     */
    private function getAdminStats()
    {
        try {
            // Usuarios totales
            $totalUsers = $this->userModel->countAllResults();
            
            // Conversaciones
            $conversationsFile = WRITEPATH . 'conversations.json';
            $conversations = file_exists($conversationsFile) ? json_decode(file_get_contents($conversationsFile), true) : [];
            $activeConversations = count(array_filter($conversations, function($conv) {
                return isset($conv['status']) && $conv['status'] === 'active';
            }));

            // Tickets
            $ticketsFile = WRITEPATH . 'tickets.json';
            $tickets = file_exists($ticketsFile) ? json_decode(file_get_contents($ticketsFile), true) : [];
            $openTickets = count(array_filter($tickets, function($ticket) {
                return $ticket['status'] === 'open';
            }));
            $urgentTickets = count(array_filter($tickets, function($ticket) {
                return $ticket['priority'] === 'urgent' && $ticket['status'] !== 'closed';
            }));

            // Flujos
            $flowsFile = WRITEPATH . 'flows.json';
            $flows = file_exists($flowsFile) ? json_decode(file_get_contents($flowsFile), true) : [];
            $activeFlows = count(array_filter($flows, function($flow) {
                return $flow['active'] === true;
            }));

            return [
                'users' => [
                    'count' => $totalUsers,
                    'label' => 'Usuarios Registrados',
                    'change' => '+' . rand(1, 5) . '%',
                    'icon' => 'fas fa-users',
                    'color' => 'primary'
                ],
                'conversations' => [
                    'count' => $activeConversations,
                    'label' => 'Conversaciones Activas',
                    'change' => '+' . rand(5, 15) . '%',
                    'icon' => 'fas fa-comments',
                    'color' => 'info'
                ],
                'tickets' => [
                    'count' => $openTickets,
                    'label' => 'Tickets Abiertos',
                    'urgent' => $urgentTickets,
                    'icon' => 'fas fa-ticket-alt',
                    'color' => 'warning'
                ],
                'flows' => [
                    'count' => $activeFlows,
                    'label' => 'Flujos Activos',
                    'icon' => 'fas fa-project-diagram',
                    'color' => 'success'
                ]
            ];

        } catch (\Exception $e) {
            return $this->getDefaultStats();
        }
    }

    /**
     * Estadísticas para asesores
     */
    private function getAdvisorStats($userId)
    {
        try {
            // Tickets asignados al asesor
            $ticketsFile = WRITEPATH . 'tickets.json';
            $tickets = file_exists($ticketsFile) ? json_decode(file_get_contents($ticketsFile), true) : [];
            
            $myTickets = array_filter($tickets, function($ticket) use ($userId) {
                return isset($ticket['assigned_to']) && $ticket['assigned_to'] == $userId;
            });

            $openTickets = count(array_filter($myTickets, function($ticket) {
                return $ticket['status'] === 'open';
            }));

            $inProgressTickets = count(array_filter($myTickets, function($ticket) {
                return $ticket['status'] === 'in_progress';
            }));

            $closedToday = count(array_filter($myTickets, function($ticket) {
                return $ticket['status'] === 'closed' && 
                       date('Y-m-d', strtotime($ticket['updated_at'] ?? '')) === date('Y-m-d');
            }));

            // Conversaciones asignadas
            $conversationsFile = WRITEPATH . 'conversations.json';
            $conversations = file_exists($conversationsFile) ? json_decode(file_get_contents($conversationsFile), true) : [];
            
            $myConversations = array_filter($conversations, function($conv) use ($userId) {
                return isset($conv['assigned_to']) && $conv['assigned_to'] == $userId;
            });

            return [
                'my_tickets' => [
                    'count' => count($myTickets),
                    'label' => 'Mis Tickets Totales',
                    'icon' => 'fas fa-ticket-alt',
                    'color' => 'primary'
                ],
                'open_tickets' => [
                    'count' => $openTickets,
                    'label' => 'Tickets Pendientes',
                    'icon' => 'fas fa-clock',
                    'color' => 'warning'
                ],
                'in_progress' => [
                    'count' => $inProgressTickets,
                    'label' => 'En Proceso',
                    'icon' => 'fas fa-cog',
                    'color' => 'info'
                ],
                'closed_today' => [
                    'count' => $closedToday,
                    'label' => 'Resueltos Hoy',
                    'icon' => 'fas fa-check-circle',
                    'color' => 'success'
                ],
                'conversations' => [
                    'count' => count($myConversations),
                    'label' => 'Conversaciones Asignadas',
                    'icon' => 'fas fa-comments',
                    'color' => 'info'
                ]
            ];

        } catch (\Exception $e) {
            return $this->getDefaultAdvisorStats();
        }
    }

    /**
     * Notificaciones para administradores
     */
    private function getAdminNotifications()
    {
        $notifications = [];

        try {
            // Tickets urgentes
            $ticketsFile = WRITEPATH . 'tickets.json';
            $tickets = file_exists($ticketsFile) ? json_decode(file_get_contents($ticketsFile), true) : [];
            
            $urgentTickets = array_filter($tickets, function($ticket) {
                return $ticket['priority'] === 'urgent' && $ticket['status'] !== 'closed';
            });

            foreach (array_slice($urgentTickets, 0, 3) as $ticket) {
                $notifications[] = [
                    'type' => 'urgent',
                    'icon' => 'fas fa-exclamation-triangle',
                    'color' => 'danger',
                    'title' => 'Ticket Urgente',
                    'message' => 'Ticket #' . $ticket['ticket_number'] . ' - ' . substr($ticket['subject'], 0, 30) . '...',
                    'time' => $this->timeAgo($ticket['created_at']),
                    'link' => base_url('admin/tickets/view/' . $ticket['id'])
                ];
            }

            // Nuevas conversaciones sin asignar
            $conversationsFile = WRITEPATH . 'conversations.json';
            $conversations = file_exists($conversationsFile) ? json_decode(file_get_contents($conversationsFile), true) : [];
            
            $unassignedConversations = array_filter($conversations, function($conv) {
                return !isset($conv['assigned_to']) || $conv['assigned_to'] === null;
            });

            if (count($unassignedConversations) > 0) {
                $notifications[] = [
                    'type' => 'info',
                    'icon' => 'fas fa-comments',
                    'color' => 'info',
                    'title' => 'Conversaciones Sin Asignar',
                    'message' => count($unassignedConversations) . ' conversaciones esperando asignación',
                    'time' => 'Ahora',
                    'link' => base_url('conversations')
                ];
            }

            // Usuarios nuevos
            $recentUsers = $this->userModel->where('created_at >', date('Y-m-d H:i:s', strtotime('-24 hours')))->findAll();
            if (count($recentUsers) > 0) {
                $notifications[] = [
                    'type' => 'success',
                    'icon' => 'fas fa-user-plus',
                    'color' => 'success',
                    'title' => 'Nuevos Usuarios',
                    'message' => count($recentUsers) . ' usuarios registrados hoy',
                    'time' => 'Hoy',
                    'link' => base_url('admin/users')
                ];
            }

        } catch (\Exception $e) {
            // Notificaciones por defecto en caso de error
        }

        return array_slice($notifications, 0, 5); // Máximo 5 notificaciones
    }

    /**
     * Notificaciones para asesores
     */
    private function getAdvisorNotifications($userId)
    {
        $notifications = [];

        try {
            // Tickets asignados recientemente
            $ticketsFile = WRITEPATH . 'tickets.json';
            $tickets = file_exists($ticketsFile) ? json_decode(file_get_contents($ticketsFile), true) : [];
            
            $recentAssignedTickets = array_filter($tickets, function($ticket) use ($userId) {
                return isset($ticket['assigned_to']) && 
                       $ticket['assigned_to'] == $userId && 
                       $ticket['status'] === 'open' &&
                       strtotime($ticket['assigned_at'] ?? '') > strtotime('-2 hours');
            });

            foreach (array_slice($recentAssignedTickets, 0, 3) as $ticket) {
                $notifications[] = [
                    'type' => 'new_ticket',
                    'icon' => 'fas fa-ticket-alt',
                    'color' => 'primary',
                    'title' => 'Nuevo Ticket Asignado',
                    'message' => 'Ticket #' . $ticket['ticket_number'] . ' - ' . substr($ticket['subject'], 0, 30) . '...',
                    'time' => $this->timeAgo($ticket['assigned_at']),
                    'link' => base_url('admin/tickets/view/' . $ticket['id'])
                ];
            }

            // Tickets urgentes asignados
            $urgentTickets = array_filter($tickets, function($ticket) use ($userId) {
                return isset($ticket['assigned_to']) && 
                       $ticket['assigned_to'] == $userId && 
                       $ticket['priority'] === 'urgent' && 
                       $ticket['status'] !== 'closed';
            });

            foreach (array_slice($urgentTickets, 0, 2) as $ticket) {
                $notifications[] = [
                    'type' => 'urgent',
                    'icon' => 'fas fa-exclamation-triangle',
                    'color' => 'danger',
                    'title' => 'Ticket Urgente',
                    'message' => 'Requiere atención inmediata - #' . $ticket['ticket_number'],
                    'time' => $this->timeAgo($ticket['created_at']),
                    'link' => base_url('admin/tickets/view/' . $ticket['id'])
                ];
            }

            // Conversaciones asignadas con mensajes nuevos
            $conversationsFile = WRITEPATH . 'conversations.json';
            $conversations = file_exists($conversationsFile) ? json_decode(file_get_contents($conversationsFile), true) : [];
            
            $myActiveConversations = array_filter($conversations, function($conv) use ($userId) {
                return isset($conv['assigned_to']) && 
                       $conv['assigned_to'] == $userId && 
                       $conv['status'] === 'active';
            });

            if (count($myActiveConversations) > 0) {
                $notifications[] = [
                    'type' => 'conversation',
                    'icon' => 'fas fa-comments',
                    'color' => 'info',
                    'title' => 'Conversaciones Activas',
                    'message' => count($myActiveConversations) . ' conversaciones esperando respuesta',
                    'time' => 'Ahora',
                    'link' => base_url('conversations')
                ];
            }

        } catch (\Exception $e) {
            // Notificaciones por defecto en caso de error
        }

        return array_slice($notifications, 0, 5); // Máximo 5 notificaciones
    }

    /**
     * Actividad reciente para administradores
     */
    private function getRecentActivity()
    {
        return [
            [
                'type' => 'ticket_created',
                'icon' => 'fas fa-ticket-alt',
                'color' => 'warning',
                'message' => 'Nuevo ticket creado por usuario',
                'time' => '5 min',
                'user' => 'Sistema'
            ],
            [
                'type' => 'user_registered',
                'icon' => 'fas fa-user-plus',
                'color' => 'success',
                'message' => 'Nuevo asesor registrado',
                'time' => '15 min',
                'user' => 'Admin'
            ],
            [
                'type' => 'conversation_transferred',
                'icon' => 'fas fa-exchange-alt',
                'color' => 'info',
                'message' => 'Conversación transferida a asesor',
                'time' => '30 min',
                'user' => 'Admin'
            ]
        ];
    }

    /**
     * Actividad reciente para asesores
     */
    private function getAdvisorActivity($userId)
    {
        return [
            [
                'type' => 'ticket_assigned',
                'icon' => 'fas fa-ticket-alt',
                'color' => 'primary',
                'message' => 'Ticket asignado a ti',
                'time' => '10 min',
                'user' => 'Admin'
            ],
            [
                'type' => 'conversation_received',
                'icon' => 'fas fa-comments',
                'color' => 'info',
                'message' => 'Nueva conversación asignada',
                'time' => '25 min',
                'user' => 'Sistema'
            ],
            [
                'type' => 'ticket_resolved',
                'icon' => 'fas fa-check-circle',
                'color' => 'success',
                'message' => 'Ticket resuelto exitosamente',
                'time' => '1 hora',
                'user' => 'Tú'
            ]
        ];
    }

    /**
     * Estadísticas por defecto en caso de error
     */
    private function getDefaultStats()
    {
        return [
            'users' => ['count' => 1, 'label' => 'Usuarios Registrados', 'change' => '+0%', 'icon' => 'fas fa-users', 'color' => 'primary'],
            'conversations' => ['count' => 0, 'label' => 'Conversaciones Activas', 'change' => '+0%', 'icon' => 'fas fa-comments', 'color' => 'info'],
            'tickets' => ['count' => 0, 'label' => 'Tickets Abiertos', 'urgent' => 0, 'icon' => 'fas fa-ticket-alt', 'color' => 'warning'],
            'flows' => ['count' => 3, 'label' => 'Flujos Activos', 'icon' => 'fas fa-project-diagram', 'color' => 'success']
        ];
    }

    /**
     * Estadísticas por defecto para asesores
     */
    private function getDefaultAdvisorStats()
    {
        return [
            'my_tickets' => ['count' => 0, 'label' => 'Mis Tickets Totales', 'icon' => 'fas fa-ticket-alt', 'color' => 'primary'],
            'open_tickets' => ['count' => 0, 'label' => 'Tickets Pendientes', 'icon' => 'fas fa-clock', 'color' => 'warning'],
            'in_progress' => ['count' => 0, 'label' => 'En Proceso', 'icon' => 'fas fa-cog', 'color' => 'info'],
            'closed_today' => ['count' => 0, 'label' => 'Resueltos Hoy', 'icon' => 'fas fa-check-circle', 'color' => 'success'],
            'conversations' => ['count' => 0, 'label' => 'Conversaciones Asignadas', 'icon' => 'fas fa-comments', 'color' => 'info']
        ];
    }

    /**
     * Función auxiliar para calcular tiempo transcurrido
     */
    private function timeAgo($datetime)
    {
        if (!$datetime) return 'Hace un momento';
        
        $time = time() - strtotime($datetime);
        
        if ($time < 60) return 'Hace ' . $time . ' seg';
        if ($time < 3600) return 'Hace ' . round($time/60) . ' min';
        if ($time < 86400) return 'Hace ' . round($time/3600) . ' h';
        return 'Hace ' . round($time/86400) . ' días';
    }

    /**
     * API para obtener estadísticas en tiempo real
     */
    public function getStatsApi()
    {
        if (!session()->get('isLoggedIn')) {
            return $this->response->setJSON(['error' => 'No autorizado']);
        }

        $role = session()->get('role');
        $userId = session()->get('user_id');

        if ($role === 'admin') {
            $stats = $this->getAdminStats();
        } else {
            $stats = $this->getAdvisorStats($userId);
        }

        return $this->response->setJSON($stats);
    }

    /**
     * API para obtener notificaciones en tiempo real
     */
    public function getNotificationsApi()
    {
        if (!session()->get('isLoggedIn')) {
            return $this->response->setJSON(['error' => 'No autorizado']);
        }

        $role = session()->get('role');
        $userId = session()->get('user_id');

        if ($role === 'admin') {
            $notifications = $this->getAdminNotifications();
        } else {
            $notifications = $this->getAdvisorNotifications($userId);
        }

        return $this->response->setJSON(['notifications' => $notifications]);
    }
}