<?php

namespace App\Controllers\Admin;

use CodeIgniter\Controller;
use App\Models\FlowModel;

class FlowController extends Controller
{
    protected $flowModel;

    public function __construct()
    {
        $this->flowModel = new FlowModel();
        
        // Verificar autenticación y rol de admin
        if (!session()->get('isLoggedIn') || session()->get('role') !== 'admin') {
            return redirect()->to(base_url('login'))->send();
        }
    }

    public function index()
    {
        // Crear flujos predeterminados si no existen
        if (!$this->flowModel->hasDefaultFlows()) {
            $this->flowModel->createDefaultFlows();
        }

        $flows = $this->flowModel->orderBy('order_index', 'ASC')->findAll();
        $stats = $this->flowModel->getFlowStats();

        return view('admin/flows/index', [
            'title' => 'Gestión de Flujos - Chat Solibot',
            'flows' => $flows,
            'stats' => $stats
        ]);
    }

    public function create()
    {
        $parentFlows = $this->flowModel->where('type', 'menu')
                                      ->orWhere('type', 'submenu')
                                      ->orderBy('name', 'ASC')
                                      ->findAll();

        return view('admin/flows/create', [
            'title' => 'Crear Flujo - Chat Solibot',
            'parentFlows' => $parentFlows
        ]);
    }

    public function store()
    {
        // Validar datos básicos
        $rules = [
            'flow_key' => 'required|max_length[100]|is_unique[flows.flow_key]',
            'name' => 'required|max_length[255]',
            'type' => 'required|in_list[menu,submenu,action,redirect]',
            'message' => 'required'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        // Procesar opciones del formulario
       // print_r($this->request->getPost('options'));die;
        $options = [];
               
            if ($this->request->getPost('options')) {
                foreach ($this->request->getPost('options') as $option) {
                    if (!empty($option['text'])) {
                        $optionData = [
                            'key' => $option['key'] ?: (count($options) + 1),
                            'text' => $option['text'],
                            'action' => $option['action'] ?: 'submenu',
                            'value' => $option['value'] ?: ''
                        ];

                        // ✅ Agregar campo response si corresponde
                        if ($optionData['action'] === 'response' && !empty($option['response'])) {
                            $optionData['response'] = $option['response'];
                        }

                        $options[] = $optionData;
                    }
                }
            }

                
                    $data = [
            'flow_key' => $this->request->getPost('flow_key'),
            'name' => $this->request->getPost('name'),
            'type' => $this->request->getPost('type'),
            'description' => $this->request->getPost('description') ?: '',
            'message' => $this->request->getPost('message'),
            'options' => $options,
            'redirect_url' => $this->request->getPost('redirect_url') ?: '',
            'parent_id' => $this->request->getPost('parent_id') ?: null,
            'order_index' => (int)($this->request->getPost('order_index') ?: 1),
            'is_active' => $this->request->getPost('is_active') ? true : false,
            'requires_advisor' => $this->request->getPost('requires_advisor') ? true : false
        ];

        if ($this->flowModel->insert($data)) {
            return redirect()->to(base_url('admin/flows'))->with('success', 'Flujo creado exitosamente');
        } else {
            return redirect()->back()->withInput()->with('errors', $this->flowModel->errors());
        }
    }

    public function edit($id)
    {
        $flow = $this->flowModel->find($id);
        
        if (!$flow) {
            return redirect()->to(base_url('admin/flows'))->with('error', 'Flujo no encontrado');
        }

        $parentFlows = $this->flowModel->where('type', 'menu')
                                      ->orWhere('type', 'submenu')
                                      ->where('id !=', $id)
                                      ->orderBy('name', 'ASC')
                                      ->findAll();

        return view('admin/flows/edit', [
            'title' => 'Editar Flujo - Chat Solibot',
            'flow' => $flow,
            'parentFlows' => $parentFlows
        ]);
    }

public function update($id)
{
    $flow = $this->flowModel->find($id);
    
    if (!$flow) {
        return redirect()->to(base_url('admin/flows'))->with('error', 'Flujo no encontrado');
    }

    // Inicializamos las reglas de validación
    $rules = [
        'flow_key' => "required|max_length[100]",
        'name' => 'required|max_length[255]',
        'type' => 'required|in_list[menu,submenu,action,redirect]',
        'message' => 'required'
    ];

    // Verificamos si el flow_key ha cambiado
    if ($this->request->getPost('flow_key') != $flow['flow_key']) {
        // Si flow_key ha cambiado, aplicar la validación `is_unique`
        $rules['flow_key'] = "required|max_length[100]|is_unique[flows.flow_key,id,{$id}]";
    }

    // Validar los datos
    if (!$this->validate($rules)) {
        return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
    }

    // Procesar las opciones del formulario
   $options = [];
        if ($this->request->getPost('options')) {
            foreach ($this->request->getPost('options') as $option) {
                if (!empty($option['text'])) {
                    $optionData = [
                        'key' => $option['key'] ?: (count($options) + 1),
                        'text' => $option['text'],
                        'action' => $option['action'] ?: 'submenu',
                        'value' => $option['value'] ?: ''
                    ];

                    // ✅ Agregar campo response si la acción lo requiere
                    if ($optionData['action'] === 'response' && !empty($option['response'])) {
                        $optionData['response'] = $option['response'];
                    }

                    $options[] = $optionData;
                }
            }
        }


    // Preparar los datos para la actualización
    $data = [
      //  'flow_key' => $this->request->getPost('flow_key'),
        'name' => $this->request->getPost('name'),
        'type' => $this->request->getPost('type'),
        'description' => $this->request->getPost('description') ?: '',
        'message' => $this->request->getPost('message'),
        'options' => $options,
        'redirect_url' => $this->request->getPost('redirect_url') ?: '',
        'parent_id' => $this->request->getPost('parent_id') ?: null,
        'order_index' => (int)($this->request->getPost('order_index') ?: 1),
        'is_active' => $this->request->getPost('is_active') ? true : false,
        'requires_advisor' => $this->request->getPost('requires_advisor') ? true : false
    ];
    
    // Intentar actualizar el flujo
    if ($this->flowModel->update($id, $data)) {
    return redirect()->to(base_url('admin/flows'))->with('success', 'Flujo actualizado exitosamente');
} else {
    // Mostrar los errores de la base de datos
    $errors = $this->flowModel->errors();  // Captura los errores
    var_dump($errors);  // Muestra los errores en el log
    die;
    return redirect()->back()->withInput()->with('errors', $errors);
}

}


    public function delete($id)
    {
        $flow = $this->flowModel->find($id);
        
        if (!$flow) {
            return redirect()->to(base_url('admin/flows'))->with('error', 'Flujo no encontrado');
        }

        // Verificar si tiene flujos hijos
        $childFlows = $this->flowModel->where('parent_id', $id)->findAll();
        if (!empty($childFlows)) {
            return redirect()->to(base_url('admin/flows'))->with('error', 'No se puede eliminar un flujo que tiene submenús asociados');
        }

        if ($this->flowModel->delete($id)) {
            return redirect()->to(base_url('admin/flows'))->with('success', 'Flujo eliminado exitosamente');
        } else {
            return redirect()->to(base_url('admin/flows'))->with('error', 'Error al eliminar el flujo');
        }
    }

    public function toggleStatus()
    {
        if (!$this->request->isAJAX()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Solicitud inválida']);
        }

        $input = $this->request->getJSON();
        $flowId = $input->flow_id ?? null;
        $isActive = $input->active ?? false;

        if (!$flowId) {
            return $this->response->setJSON(['success' => false, 'message' => 'ID de flujo requerido']);
        }

        $flow = $this->flowModel->find($flowId);
        if (!$flow) {
            return $this->response->setJSON(['success' => false, 'message' => 'Flujo no encontrado']);
        }

        if ($this->flowModel->update($flowId, ['is_active' => $isActive])) {
            return $this->response->setJSON(['success' => true]);
        } else {
            return $this->response->setJSON(['success' => false, 'message' => 'Error al actualizar el estado']);
        }
    }

    public function duplicate($id)
    {
        $flow = $this->flowModel->find($id);
        
        if (!$flow) {
            return redirect()->to(base_url('admin/flows'))->with('error', 'Flujo no encontrado');
        }

        // Crear copia del flujo
        unset($flow['id']);
        $flow['flow_key'] = $flow['flow_key'] . '_copy_' . time();
        $flow['name'] = $flow['name'] . ' (Copia)';
        $flow['is_active'] = false; // Crear inactivo por defecto

        if ($this->flowModel->insert($flow)) {
            return redirect()->to(base_url('admin/flows'))->with('success', 'Flujo duplicado exitosamente');
        } else {
            return redirect()->to(base_url('admin/flows'))->with('error', 'Error al duplicar el flujo');
        }
    }

    /**
     * API para obtener flujos por tipo
     */
    public function getFlowsByType($type)
    {
        if (!$this->request->isAJAX()) {
            return $this->response->setJSON(['error' => 'Solicitud inválida']);
        }

        $flows = $this->flowModel->where('type', $type)
                                ->where('is_active', true)
                                ->orderBy('order_index', 'ASC')
                                ->findAll();

        return $this->response->setJSON(['flows' => $flows]);
    }

    /**
     * API para obtener estadísticas
     */
    public function getStatsApi()
    {
        if (!$this->request->isAJAX()) {
            return $this->response->setJSON(['error' => 'Solicitud inválida']);
        }

        $stats = $this->flowModel->getFlowStats();
        return $this->response->setJSON($stats);
    }
}