<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\UserModel;
use Sonata\GoogleAuthenticator\GoogleAuthenticator;
use Sonata\GoogleAuthenticator\GoogleQrUrl;

class AdvisorController extends BaseController
{
    protected $userModel;

    public function __construct()
    {
        $this->userModel = new UserModel();
    }

    /**
     * Mostrar perfil del asesor
     */
    public function profile()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to(base_url('login'));
        }

        $userId = session()->get('userId');
        $user = $this->userModel->find($userId);

        if (!$user) {
            return redirect()->to(base_url('login'))->with('error', 'Usuario no encontrado');
        }

        return view('advisor/profile', [
            'title' => 'Mi Perfil - Chat Solibot',
            'user' => $user
        ]);
    }

    /**
     * Actualizar información del perfil
     */
    public function updateProfile()
    {
        $userId = session()->get('userId');
      
        $rules = [
            'name' => 'required|max_length[100]', 
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'name' => $this->request->getPost('name'),
        ];   

        if ($this->userModel->update($userId, $data)) {
            return redirect()->to(base_url('advisor/profile'))->with('success', 'Perfil actualizado exitosamente');
        } else {
            return redirect()->back()->with('error', 'Error al actualizar el perfil');
        }
    }

    /**
     * Cambiar contraseña
     */
    public function changePassword()
    {
        $userId = session()->get('userId');
        $user = $this->userModel->find($userId);

        $currentPassword = $this->request->getPost('current_password');
        $newPassword = $this->request->getPost('new_password');
        $confirmPassword = $this->request->getPost('confirm_password');

        if (!password_verify($currentPassword, $user['password'])) {
            return redirect()->back()->with('error', 'La contraseña actual es incorrecta');
        }

        if (strlen($newPassword) < 6) {
            return redirect()->back()->with('error', 'La nueva contraseña debe tener al menos 6 caracteres');
        }

        if ($newPassword !== $confirmPassword) {
            return redirect()->back()->with('error', 'Las contraseñas no coinciden');
        }

        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        
        if ($this->userModel->update($userId, ['password' => $hashedPassword])) {
            return redirect()->to(base_url('advisor/profile'))->with('success', 'Contraseña cambiada exitosamente');
        } else {
            return redirect()->back()->with('error', 'Error al cambiar la contraseña');
        }
    }

    /**
     * Configurar 2FA por primera vez
     */
    public function setup2FA()
    {
        $userId = session()->get('userId');
        $user = $this->userModel->find($userId);

        // Si ya tiene 2FA configurado, redirigir
        if (!empty($user['google_auth_secret'])) {
            return redirect()->to(base_url('advisor/profile'))->with('info', 'La autenticación de dos factores ya está configurada');
        }

        $ga = new GoogleAuthenticator();
        $secret = $ga->generateSecret();

        // Guardar temporalmente el secreto en sesión
        session()->set('temp_2fa_secret', $secret);

        // Generar QR
        $qrCodeUrl = GoogleQrUrl::generate('Chat Solibot (' . $user['username'] . ')', $secret);

        return view('advisor/setup_2fa', [
            'title' => 'Configurar Autenticación de Dos Factores',
            'qrCodeUrl' => $qrCodeUrl,
            'secret' => $secret,
            'user' => $user
        ]);
    }

    /**
     * Verificar y activar 2FA
     */
    public function verify2FASetup()
    {
        $userId = session()->get('userId');
        $code = $this->request->getPost('verification_code');
        $secret = session()->get('temp_2fa_secret');

        if (!$secret) {
            return redirect()->to(base_url('advisor/profile'))->with('error', 'Sesión expirada. Inicia la configuración nuevamente');
        }

        $ga = new GoogleAuthenticator();
        
        if ($ga->checkCode($secret, $code)) {
            // Código correcto, activar 2FA
            $this->userModel->update($userId, [
                'google_auth_secret' => $secret,
                'two_factor_enabled' => true
            ]);

            // Limpiar sesión temporal
            session()->remove('temp_2fa_secret');

            return redirect()->to(base_url('advisor/profile'))->with('success', 'Autenticación de dos factores activada exitosamente');
        } else {
            return redirect()->back()->with('error', 'Código de verificación incorrecto');
        }
    }

    /**
     * Toggle 2FA (activar/desactivar) - MÉTODO PRINCIPAL
     */
    public function toggle2FA()
    {
        $userId = session()->get('userId');
        $user = $this->userModel->find($userId);
        $action = $this->request->getPost('action'); // 'enable' o 'disable'
//print_r($userId);die;
        if ($action === 'enable') {
                $this->userModel->update($userId, ['two_factor_enabled' => true]);
                return redirect()->to(base_url('advisor/profile'))->with('success', 'Autenticación de dos factores activada');
            
        } elseif ($action === 'disable') {
            // Desactivar 2FA (mantener el secreto)
            $this->userModel->update($userId, ['two_factor_enabled' => false]);
            return redirect()->to(base_url('advisor/profile'))->with('success', 'Autenticación de dos factores desactivada');
        }

        return redirect()->to(base_url('advisor/profile'))->with('error', 'Acción no válida');
    }

    /**
     * Desactivar 2FA completamente (eliminar secreto)
     */
    public function disable2FA()
    {
        $userId = session()->get('userId');
        $user = $this->userModel->find($userId);
        $password = $this->request->getPost('password');
        $code = $this->request->getPost('verification_code');

        // Verificar contraseña
        if (!password_verify($password, $user['password'])) {
            return redirect()->back()->with('error', 'Contraseña incorrecta');
        }

        // Verificar código 2FA actual si está activado
        if (!empty($user['google_auth_secret']) && $user['two_factor_enabled']) {
            $ga = new GoogleAuthenticator();
            if (!$ga->checkCode($user['google_auth_secret'], $code)) {
                return redirect()->back()->with('error', 'Código de verificación incorrecto');
            }
        }

        // Desactivar y eliminar 2FA completamente
        $this->userModel->update($userId, [
            'google_auth_secret' => null,
            'two_factor_enabled' => false
        ]);

        return redirect()->to(base_url('advisor/profile'))->with('success', 'Autenticación de dos factores eliminada completamente');
    }
}