<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\FlowModel;
use App\Models\AdvisorAssignmentModel;
use App\Models\UserModel;

class ConversationController extends BaseController
{
    protected $flowModel;
     protected $advisorAssignmentModel;
     protected $userModel;
     protected $db; // ✅ Necesario si estás en modo estricto


    public function __construct()
    {
        $this->flowModel = new FlowModel();
        $this->advisorAssignmentModel = new AdvisorAssignmentModel();
          $this->userModel = new UserModel();
           $this->db = \Config\Database::connect(); 
    }

public function index($sender = null)
{
    $file = WRITEPATH . 'conversations.json';
    $ticketsFile = WRITEPATH . 'tickets.json';

    $currentUserId = session()->get('userId');
    $role = session()->get('role');

    $conversations = file_exists($file) ? json_decode(file_get_contents($file), true) : [];
    $tickets = file_exists($ticketsFile) ? json_decode(file_get_contents($ticketsFile), true) : [];

    $filteredConversations = [];
    $resultado = [];

    foreach ($conversations as $userPhone => $data) {
        // ❌ Ignorar conversaciones sin mensajes
        if (empty($data['messages']) || !is_array($data['messages'])) {
            continue;
        }

        $lastNewFlow = null;
        $flowId = null;

        // Buscar último mensaje del bot con new_flow y flow_id
        foreach (array_reverse($data['messages']) as $msg) {
            if (
                isset($msg['sender_type'], $msg['new_flow'], $msg['flow_id']) &&
                $msg['sender_type'] === 'bot'
            ) {
                $lastNewFlow = $msg['new_flow'];
                $flowId = $msg['flow_id'];
                break;
            }
        }

        // Alternativa si no se encontró new_flow
        if (!$lastNewFlow && !$flowId && isset($data['current_flow'])) {
            foreach (array_reverse($data['messages']) as $msg) {
                if (
                    isset($msg['sender_type'], $msg['flow_id']) &&
                    $msg['sender_type'] === 'bot'
                ) {
                    $lastNewFlow = $data['current_flow'];
                    $flowId = $msg['flow_id'];
                    break;
                }
            }
        }

        // Verificar si tiene ticket ABIERTO asignado al asesor
        $hasOpenTicket = false;
        foreach ($tickets as $ticket) {
       
            if (isset($ticket['user_phone'], $ticket['status'], $ticket['assigned_to']) &&
                $ticket['user_phone'] == $userPhone &&
                $ticket['status'] !== 'closed' &&
                $ticket['assigned_to'] == $currentUserId
            ) {
                  
                $hasOpenTicket = true;
                break;
            }
        }
 
   
        $asignados = [];
        if ($lastNewFlow && $flowId) {
            $asignados = array_column($this->advisorAssignmentModel->getAdvisorsByFlow($flowId, $lastNewFlow), 'user_id');
        }

        $resultado[$userPhone] = [
            'flow_id' => $flowId,
            'flow_key' => $lastNewFlow,
            'asesores' => $asignados,
        ];

        $canAccess = false;
        if ($role === 'admin') {
            $canAccess = true;
        } else {
            $canAccess = in_array($currentUserId, $asignados) || $hasOpenTicket;
        }

        if ($canAccess) {
            $filteredConversations[$userPhone] = $data;
            $filteredConversations[$userPhone]['asignados'] = $asignados;
            $filteredConversations[$userPhone]['access_reason'] =
                $hasOpenTicket ? 'ticket' :
                (in_array($currentUserId, $asignados) ? 'flow' : 'unknown');
        }
    }

    // ✅ NO agregar conversaciones vacías virtuales
    // Ya que deben tener al menos un mensaje para mostrarse
    // Así evitamos mostrar "0 mensajes" como en tu imagen

    // Obtener conversación seleccionada
    if (!empty($filteredConversations) && $sender && isset($filteredConversations[$sender])) {
        $selectedSender = $sender;
        $selectedConversation = $filteredConversations[$sender]['messages'] ?? [];
    } else {
        $selectedSender = null;
        $selectedConversation = [];
    }

    return view('conversations/index', [
        'conversations' => $filteredConversations,
        'selectedSender' => $selectedSender,
        'selectedConversation' => $selectedConversation,
        'resultado' => $resultado,
    ]);
}



  public function webhook()
{
    try {
         $this->checkUserInactivity();
        $input = $this->request->getJSON();

        // Log del webhook recibido (opcional)
        file_put_contents(WRITEPATH . 'webhook_log.txt',
            "Webhook recibido: " . json_encode($input, JSON_PRETTY_PRINT) . "\n", FILE_APPEND);

        if (!isset($input->entry[0]->changes[0]->value->messages)) {
            // Si no hay mensajes, responder con éxito sin hacer nada
            return $this->response->setJSON(['status' => 'no messages'])->setStatusCode(200);
        }

        $messages = $input->entry[0]->changes[0]->value->messages;
        $metadata = $input->entry[0]->changes[0]->value->metadata ?? null;
        $botNumber = $metadata->display_phone_number ?? null;

        $file = WRITEPATH . 'conversations.json';
        $conversations = file_exists($file) ? json_decode(file_get_contents($file), true) : [];

        // Archivo para guardar IDs de mensajes procesados
        $logFile = WRITEPATH . 'processed_messages.log';
        $processed = file_exists($logFile) ? explode("\n", trim(file_get_contents($logFile))) : [];

        foreach ($messages as $message) {

            // ⚠️ Ignorar si el mensaje no es texto o no tiene ID
            if (!isset($message->text) || !isset($message->id)) {
                continue;
            }

            // ⚠️ Ignorar mensajes enviados por el bot
            if ($message->from === $botNumber) {
                continue;
            }

            // ⚠️ Ignorar si el mensaje ya fue procesado
            $messageId = $message->id;
            if (in_array($messageId, $processed)) {
                continue;
            }

            // ✅ Marcar como procesado
            file_put_contents($logFile, $messageId . "\n", FILE_APPEND);

            $from = $message->from;
            $userMessage = $message->text->body ?? '';

            // Inicializar conversación si no existe
            if (!isset($conversations[$from])) {
                 
                $conversations[$from] = [
                    'user_phone' => $from,
                    'status' => 'active',
                    'assigned_to' => null,
                    'created_at' => time(),
                    'last_activity' => time(),
                    'current_flow' => null,
                    'flow_context' => [],
                    'messages' => []
                ];

                $welcomeFlow = $this->flowModel->getFlowByKey('main_menu');

                if ($welcomeFlow) {
                    $welcomeMessage = $this->flowModel->formatFlowForWhatsApp($welcomeFlow);

                    $conversations[$from]['messages'][] = [
                        'message' => $welcomeMessage,
                        'type' => 'sent',
                        'sender_type' => 'bot',
                        'timestamp' => time(),
                        'flow_id' => $welcomeFlow['id']
                    ];
                    $conversations[$from]['current_flow'] = 'main_menu';

                    $this->sendToWhatsApp($from, $welcomeMessage);
                } else {
                    $welcomeMessage = "¡Hola! Bienvenido a nuestro servicio. Por favor, selecciona una opción:\n\n1. Soporte técnico\n2. Productos\n3. Otros";

                    $conversations[$from]['messages'][] = [
                        'message' => $welcomeMessage,
                        'type' => 'sent',
                        'sender_type' => 'bot',
                        'timestamp' => time(),
                    ];
                    $this->sendToWhatsApp($from, $welcomeMessage);
                }
            } else {
               
                $conversations[$from]['last_activity'] = time();

                $conversations[$from]['messages'][] = [
                    'message' => $userMessage,
                    'type' => 'received',
                    'sender_type' => 'user',
                    'timestamp' => time(),
                ];

                if (!$conversations[$from]['assigned_to'] && !empty(trim($userMessage))||trim($userMessage)==0) {
                  
                    $response = $this->processUserMessage($userMessage, $conversations[$from]);
                     
                    if ($response && !empty($response['message'])) {
                         sleep(1);
                        $conversations[$from]['messages'][] = [
                            'message' => $response['message'],
                            'type' => 'sent',
                            'sender_type' => 'bot',
                            'timestamp' => time(),
                            'flow_id' => $response['flow_id'] ?? null,
                            'new_flow' => $response['new_flow'] ?? null
                            
                        ];

                        if (isset($response['new_flow'])) {
                            $conversations[$from]['current_flow'] = $response['new_flow'];
                        }

                        $this->sendToWhatsApp($from, $response['message']);
                    }
                }
            }
        }

        file_put_contents($file, json_encode($conversations, JSON_PRETTY_PRINT));
        return $this->response->setJSON(['status' => 'success'])->setStatusCode(200);

    } catch (\Exception $e) {
        file_put_contents(WRITEPATH . 'webhook_error.txt',
            "Error: " . $e->getMessage() . "\n" . $e->getTraceAsString() . "\n", FILE_APPEND);

        return $this->response->setJSON(['status' => 'error', 'message' => $e->getMessage()]);
    }
}


    // Procesar mensaje del usuario basado en flujos de BD
    private function processUserMessage($userMessage, $conversationData)
    {
        $userMessage = trim($userMessage);
        
        // Si el mensaje está vacío, no procesar
        if (empty($userMessage || $userMessage==0)) {
            return null;
        }

        $currentFlow = $conversationData['current_flow'] ?? 'main_menu';

        // Si el usuario escribe "menu" o "inicio", volver al menú principal
        if (in_array(strtolower($userMessage), ['menu', 'inicio', 'principal', 'volver', '0'])) {
            $mainFlow = $this->flowModel->getFlowByKey('main_menu');
            if ($mainFlow) {
                return [
                    'message' => $this->flowModel->formatFlowForWhatsApp($mainFlow),
                    'flow_id' => $mainFlow['id'],
                    'new_flow' => 'main_menu'
                ];
            }
        }
 
        // Obtener el flujo actual desde la BD
        $flow = $this->flowModel->getFlowByKey($currentFlow);
        if (!$flow || !$flow['options']) {
            // Si no hay flujo actual, mostrar menú principal
            $mainFlow = $this->flowModel->getFlowByKey('main_menu');
            if ($mainFlow) {
                return [
                    'message' => $this->flowModel->formatFlowForWhatsApp($mainFlow),
                    'flow_id' => $mainFlow['id'],
                    'new_flow' => 'main_menu'
                ];
            }
            // Fallback al procesamiento original
            return ['message' => $this->processMessage($userMessage), 'flow_id' => null];
        }

        // Buscar la opción seleccionada por el usuario
        $options = is_string($flow['options']) ? json_decode($flow['options'], true) : $flow['options'];
        $selectedOption = null;

        // CORREGIDO: Acceder como array, no como objeto
        foreach ($options as $option) {
             
            if (isset($option->key) && $option->key === $userMessage) {
                $selectedOption = $option;
                break;
            }
        }

        if (!$selectedOption) {
            // Opción no válida, mostrar mensaje de error y repetir opciones
            $errorMessage = "❌ Opción no válida. Por favor, selecciona una de las siguientes opciones:\n\n";
            $errorMessage .= $this->flowModel->formatFlowForWhatsApp($flow);
            return [
                'message' => $errorMessage,
                'flow_id' => $flow['id'],
                'new_flow' => $currentFlow
            ];
        }

        // Procesar la acción seleccionada
        return $this->processFlowAction($selectedOption, $flow);
    }

    // Procesar acción del flujo
   private function processFlowAction($option, $currentFlow)
{

   
    switch ($option->action) {

           case 'response':

            // ✅ NUEVO: Manejar respuestas directas

            $responseMessage = $option->response ?? 'Respuesta no configurada';
            // Agregar opciones adicionales al final de la respuesta
            $responseMessage .= "\n\n¿Necesitas algo más?\n\n";
            $responseMessage .= "Escribe *menu* para volver al menú principal o *0* para hablar con un asesor.";
            return [

                'message' => $responseMessage,

                'flow_id' => $currentFlow['id'],

                'new_flow' => $currentFlow['flow_key']

            ];
            
        case 'submenu':
            $targetFlowKey = $option->value;
            $targetFlow = $this->flowModel->getFlowByKey($targetFlowKey);

            if ($targetFlow) {
                return [
                    'message' => $this->flowModel->formatFlowForWhatsApp($targetFlow),
                    'flow_id' => $targetFlow['id'],
                    'new_flow' => $targetFlow['flow_key']
                ];
            } else {
                return [
                    'message' => "❌ Lo siento, esta opción no está disponible en este momento.",
                    'flow_id' => $currentFlow['id'],
                    'new_flow' => $currentFlow['flow_key']
                ];
            }

        case 'url':
            $url = $option->value;
            $message = "📎 *{$option->text}*\n\n";
            $message .= "Puedes acceder al documento en el siguiente enlace:\n";
            $message .= "🔗 {$url}\n\n";
            $message .= "¿Necesitas algo más?\n\n";
            $message .= "Escribe *menu* para volver al menú principal o *0* para hablar con un asesor.";

            return [
                'message' => $message,
                'flow_id' => $currentFlow['id'],
                'new_flow' => $currentFlow['flow_key']
            ];

        case 'action':
            return $this->executeSpecificAction($option->value, $option->text, $currentFlow);

        case 'advisor':
            $message = "👨‍💼 *Conectando con un asesor...*\n\n";
            $message .= "En breve uno de nuestros asesores especializados se pondrá en contacto contigo para brindarte la mejor atención.\n\n";
            $message .= "⏰ *Horario de atención:*\n";
            $message .= "Lunes a Viernes: 8:00 AM - 6:00 PM\n";
            $message .= "Sábados: 8:00 AM - 12:00 PM\n\n";
            $message .= "Mientras tanto, puedes escribir *menu* para volver al menú principal.";

            $this->markForAdvisorAssignment($currentFlow);

            return [
                'message' => $message,
                'flow_id' => $currentFlow['id'],
                'new_flow' => 'advisor_requested'
            ];

        default:
            return [
                'message' => "❌ Acción no reconocida. Escribe *menu* para volver al menú principal.",
                'flow_id' => $currentFlow['id'],
                'new_flow' => $currentFlow['flow_key']
            ];
    }
}

    // Ejecutar acciones específicas
    private function executeSpecificAction($actionValue, $actionText, $currentFlow)
    {

         file_put_contents(WRITEPATH . 'yuri.txt', json_encode($actionValue, JSON_PRETTY_PRINT) . "\n\n", FILE_APPEND);

        switch ($actionValue) {
            case 'solicitud_credito':
                $message = "📝 *Solicitud de Crédito*\n\n";
                $message .= "Para radicar tu solicitud de crédito, necesitamos que nos proporciones la siguiente información:\n\n";
                $message .= "📋 *Documentos requeridos:*\n";
                $message .= "• Cédula de ciudadanía\n";
                $message .= "• Certificado de ingresos\n";
                $message .= "• Referencias comerciales\n\n";
                $message .= "👨‍💼 Te conectaremos con un asesor especializado para completar tu solicitud.\n\n";
                $message .= "Escribe *menu* para volver al menú principal.";
                break;

            case 'consultar_saldo':
                $message = "💵 *Consulta de Saldo*\n\n";
                $message .= "Para consultar tu saldo actual, puedes:\n\n";
                $message .= "📱 *Opciones disponibles:*\n";
                $message .= "• Ingresar a nuestra página web\n";
                $message .= "• Llamar a la línea de atención\n";
                $message .= "• Visitar cualquiera de nuestras oficinas\n\n";
                $message .= "👨‍💼 ¿Necesitas ayuda personalizada? Escribe *0* para hablar con un asesor.\n\n";
                $message .= "Escribe *menu* para volver al menú principal.";
                break;

            case 'info_cdat':
                $message = "💎 *Certificados de Depósito a Término (CDAT)*\n\n";
                $message .= "Los CDAT son una excelente opción de inversión que te ofrece:\n\n";
                $message .= "✅ *Beneficios:*\n";
                $message .= "• Rentabilidad garantizada\n";
                $message .= "• Diferentes plazos disponibles\n";
                $message .= "• Seguridad en tu inversión\n";
                $message .= "• Tasas competitivas\n\n";
                $message .= "👨‍💼 Para más información y asesoría personalizada, escribe *0*.\n\n";
                $message .= "Escribe *menu* para volver al menú principal.";
                break;

            case 'estado_auxilios':
                $message = "🔍 *Estado de Solicitudes de Auxilios*\n\n";
                $message .= "Para consultar el estado de tus solicitudes de auxilios:\n\n";
                $message .= "📋 *Información necesaria:*\n";
                $message .= "• Número de solicitud\n";
                $message .= "• Cédula de ciudadanía\n";
                $message .= "• Tipo de auxilio solicitado\n\n";
                $message .= "👨‍💼 Un asesor te ayudará con la consulta. Escribe *0* para conectarte.\n\n";
                $message .= "Escribe *menu* para volver al menú principal.";
                break;

            case 'solicitud_libre':
                $message = "✍️ *Solicitud Específica*\n\n";
                $message .= "Por favor, describe detalladamente tu solicitud o consulta.\n\n";
                $message .= "📝 *Incluye:*\n";
                $message .= "• Tipo de servicio que necesitas\n";
                $message .= "• Detalles específicos\n";
                $message .= "• Cualquier información relevante\n\n";
                $message .= "👨‍💼 Un asesor revisará tu solicitud y te contactará pronto.\n\n";
                $message .= "Escribe *menu* para volver al menú principal.";
                break;

            case 'info_general':
                $message = "ℹ️ *Información General*\n\n";
                $message .= "🏦 *Sobre nosotros:*\n";
                $message .= "Somos una institución financiera comprometida con el bienestar de nuestros asociados.\n\n";
                $message .= "📍 *Oficinas:* Consulta nuestras ubicaciones en la página web\n";
                $message .= "📞 *Línea de atención:* 01-8000-123-456\n";
                $message .= "🌐 *Página web:* www.ejemplo.com\n\n";
                $message .= "👨‍💼 Para atención personalizada, escribe *0*.\n\n";
                $message .= "Escribe *menu* para volver al menú principal.";
                break;

            default:
                $message = "🤖 *{$actionText}*\n\n";
                $message .= "Esta funcionalidad está en desarrollo. Pronto estará disponible.\n\n";
                $message .= "👨‍💼 Mientras tanto, puedes hablar con un asesor escribiendo *0*.\n\n";
                $message .= "Escribe *menu* para volver al menú principal.";
        }

        return [
            'message' => $message,
            'flow_id' => $currentFlow['id'],
            'new_flow' => $currentFlow['flow_key']
        ];
    }

    // Método para procesar respuestas enviadas desde el asesor
    public function send()
    {
        try {
            $input = $this->request->getJSON();
    
            $file = WRITEPATH . 'conversations.json';
            $conversations = file_exists($file) ? json_decode(file_get_contents($file), true) : [];
    
            if (!empty($input->sender) && !empty($input->message)) {
                // Agregar el mensaje del asesor a la conversación
                $conversations[$input->sender]['messages'][] = [
                    'message' => $input->message,
                    'type' => 'sent',
                    'sender_type' => 'advisor',
                    'timestamp' => time(),
                ];

                $conversations[$input->sender]['last_activity'] = time();
    
                // Guardar la conversación actualizada
                file_put_contents($file, json_encode($conversations, JSON_PRETTY_PRINT));
    
                // Enviar el mensaje al cliente a través de WhatsApp
                $this->sendToWhatsApp($input->sender, $input->message);
    
                return $this->response->setJSON(['status' => 'success']);
            } else {
                return $this->response->setJSON(['status' => 'error', 'message' => 'Datos inválidos']);
            }
        } catch (\Exception $e) {
            // Registrar errores en logs
            file_put_contents(WRITEPATH . 'send_error.txt', $e->getMessage(), FILE_APPEND);
            return $this->response->setJSON(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    // Método original mantenido como fallback
    private function processMessage($userMessage)
    {
        // Normalizar el mensaje para evitar problemas de formato
        $userMessage = strtolower(trim($userMessage));
    
        // Procesar la selección del usuario
        switch ($userMessage) {
            case '1':
                return 'Has seleccionado Soporte técnico. Un asesor te atenderá en breve.';
            case '2':
                return 'Has seleccionado Información de productos. Un asesor de ventas te contactará pronto.';
            case '3':
                return 'Por favor, indícanos más detalles sobre tu consulta.';
            default:
                return 'Lo siento, no entendí tu selección. Por favor, responde con 1, 2 o 3.';
        }
    }

    private function sendToWhatsApp($to, $message)
    {
       // $api_url = "https://waba-sandbox.360dialog.io/v1/messages";
       // $api_key = '4RvS4C_sandbox';

        $api_url = "https://waba-v2.360dialog.io/messages";
        $api_key = 'gIWJVuWLsASO4o9nfQGV392EAK';


        $data = [
            "messaging_product" => "whatsapp",
            "recipient_type" => "individual",
            "to" => $to,
            "type" => "text",
            "text" => ["body" => $message],
        ];

        $ch = curl_init($api_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Content-Type: application/json",
            "D360-API-KEY: $api_key",
        ]);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));

        $response = curl_exec($ch);

        if (curl_errno($ch)) {
            file_put_contents(WRITEPATH . 'whatsapp_error.txt', "Error cURL: " . curl_error($ch) . "\n", FILE_APPEND);
        }

        curl_close($ch);
        return json_decode($response, true);
    }

    public function getConversation($sender)
    {
        $file = WRITEPATH . 'conversations.json';
        $conversations = file_exists($file) ? json_decode(file_get_contents($file), true) : [];

        if (isset($conversations[$sender])) {
            
            return $this->response->setJSON([
                'status' => 'success',
                'conversation' => $conversations[$sender]['messages'] ?? []
            ]);
        }

        return $this->response->setJSON(['status' => 'error', 'message' => 'No hay mensajes para este remitente.']);
    }


    public function refreshConversationList()
{
    $file = WRITEPATH . 'conversations.json';
    $ticketsFile = WRITEPATH . 'tickets.json';

    $currentUserId = session()->get('userId');
    $role = session()->get('role');
    $lastTimestamp = $this->request->getGet('last_timestamp') ?? 0;
    $selectedSender = $this->request->getGet('selected') ?? '';

    $conversations = file_exists($file) ? json_decode(file_get_contents($file), true) : [];
    $tickets = file_exists($ticketsFile) ? json_decode(file_get_contents($ticketsFile), true) : [];

    $filteredConversations = [];
    $maxTimestamp = 0;

    foreach ($conversations as $userPhone => $data) {
        // ❌ Ignorar si no hay mensajes reales
        if (!isset($data['messages']) || !is_array($data['messages']) || empty($data['messages'])) {
            continue;
        }

        $maxTimestamp = max($maxTimestamp, $data['last_activity'] ?? 0);

        $lastNewFlow = null;
        $flowId = null;

        // Buscar último mensaje del bot con flow info
        foreach (array_reverse($data['messages']) as $msg) {
            if (
                isset($msg['sender_type'], $msg['new_flow'], $msg['flow_id']) &&
                $msg['sender_type'] === 'bot'
            ) {
                $lastNewFlow = $msg['new_flow'];
                $flowId = $msg['flow_id'];
                break;
            }
        }

        if (!$lastNewFlow && !$flowId && isset($data['current_flow'])) {
            foreach (array_reverse($data['messages']) as $msg) {
                if (
                    isset($msg['sender_type'], $msg['flow_id']) &&
                    $msg['sender_type'] === 'bot'
                ) {
                    $lastNewFlow = $data['current_flow'];
                    $flowId = $msg['flow_id'];
                    break;
                }
            }
        }

        // Verificar si hay un ticket abierto asignado
        $hasOpenTicket = false;
        foreach ($tickets as $ticket) {
            if (
                isset($ticket['user_phone'], $ticket['status'], $ticket['assigned_to']) &&
                $ticket['user_phone'] == $userPhone &&
                $ticket['status'] !== 'closed' &&
                $ticket['assigned_to'] == $currentUserId
            ) {
                $hasOpenTicket = true;
                break;
            }
        }

        $asignados = [];
        if ($lastNewFlow && $flowId) {
            $asignados = array_column($this->advisorAssignmentModel->getAdvisorsByFlow($flowId, $lastNewFlow), 'user_id');
        }

        $canAccess = false;
        if ($role === 'admin') {
            $canAccess = true;
        } else {
            $canAccess = in_array($currentUserId, $asignados) || $hasOpenTicket;
        }

        if ($canAccess) {
            $filteredConversations[$userPhone] = $data;
            $filteredConversations[$userPhone]['asignados'] = $asignados;
            $filteredConversations[$userPhone]['access_reason'] =
                $hasOpenTicket ? 'ticket' :
                (in_array($currentUserId, $asignados) ? 'flow' : 'new_contact');
        }
    }

    // ✅ NO agregar conversaciones vacías por ticket (NO queremos 0 mensajes)
    // Por eso esta parte se elimina

    if ((int)$maxTimestamp <= (int)$lastTimestamp) {
        return $this->response->setJSON([
            'status' => 'no-changes',
            'last_timestamp' => $maxTimestamp
        ]);
    }

    // 🖼️ Generar HTML
    $html = '';
    foreach ($filteredConversations as $sender => $conversation) {
        $lastMessage = end($conversation['messages'])['message'] ?? 'Sin mensajes';
        $html .= '
        <div class="conversation-item ' . ($sender === $selectedSender ? 'active' : '') . '" data-sender="' . $sender . '">
            <div class="conversation-header">
                <div class="conversation-name">' . htmlspecialchars($sender) . '</div>
                <div class="conversation-time">' . date('H:i', $conversation['last_activity'] ?? time()) . '</div>
            </div>
            <div class="conversation-preview">' . htmlspecialchars(mb_substr($lastMessage, 0, 50)) . '</div>
            <div class="conversation-status">
                <span class="status-badge status-bot">Bot Automático</span>
                <small class="text-muted">' . count($conversation['messages'] ?? []) . ' mensajes</small>
            </div>
        </div>';
    }

    return $this->response->setJSON([
        'status' => 'success',
        'html' => $html,
        'last_timestamp' => $maxTimestamp
    ]);
}


// 🆕 NUEVA FUNCIÓN: Tomar conversación
    public function takeControl()
    {
       // print_r($this->request->getJSON());die;
        try {
            $input = $this->request->getJSON();
            $sender = $input->sender ?? '';
            $currentUserId = session()->get('userId');
            $userName = session()->get('name') ?? 'Asesor';

            if (empty($sender) || empty($currentUserId)) {
                return $this->response->setJSON([
                    'status' => 'error', 
                    'message' => 'Datos inválidos'
                ]);
            }

            $file = WRITEPATH . 'conversations.json';
            $conversations = file_exists($file) ? json_decode(file_get_contents($file), true) : [];

            if (!isset($conversations[$sender])) {
                return $this->response->setJSON([
                    'status' => 'error', 
                    'message' => 'Conversación no encontrada'
                ]);
            }

            // Verificar si ya está tomada por otro asesor
          if (isset($conversations[$sender]['is_taken']) && 
                $conversations[$sender]['is_taken'] && 
                $conversations[$sender]['taken_by'] != $currentUserId) {

                return $this->response->setJSON([
                    'status' => 'error', 
                    'message' => 'Esta conversación ya fue tomada por otro asesor'
                ]);
            }

            // Marcar como tomada
            $conversations[$sender]['is_taken'] = true;
            $conversations[$sender]['taken_by'] = $currentUserId;
            $conversations[$sender]['taken_at'] = time();
            $conversations[$sender]['assigned_to'] = $currentUserId;

            // Agregar mensaje del sistema
            $systemMessage = "🤖➡️👨‍💼 *{$userName}* ha tomado esta conversación. Las respuestas automáticas han sido desactivadas.";
            
            $conversations[$sender]['messages'][] = [
                'message' => $systemMessage,
                'type' => 'system',
                'sender_type' => 'system',
                'timestamp' => time(),
            ];

            // Guardar cambios
            file_put_contents($file, json_encode($conversations, JSON_PRETTY_PRINT));

            // Enviar notificación al cliente
            $clientMessage = "👨‍💼 Hola, soy *{$userName}*, tu asesor personal. Ahora te atenderé directamente para brindarte la mejor ayuda posible.";
            $this->sendToWhatsApp($sender, $clientMessage);

            return $this->response->setJSON([
                'status' => 'success',
                'message' => 'Conversación tomada exitosamente'
            ]);

        } catch (\Exception $e) {
            file_put_contents(WRITEPATH . 'take_conversation_error.txt', $e->getMessage(), FILE_APPEND);
            return $this->response->setJSON([
                'status' => 'error', 
                'message' => 'Error interno del servidor'
            ]);
        }
    }


    public function getConversationStatus($sender)
{
    $file = WRITEPATH . 'conversations.json';
    $conversations = file_exists($file) ? json_decode(file_get_contents($file), true) : [];

    if (!isset($conversations[$sender])) {
        return $this->response->setJSON([
            'status' => 'error',
            'message' => 'Conversación no encontrada'
        ]);
    }

    return $this->response->setJSON([
        'status' => 'success',
        'is_taken' => $conversations[$sender]['is_taken'] ?? false,
        'taken_by' => $conversations[$sender]['taken_by'] ?? null,
        'assigned_to' => $conversations[$sender]['assigned_to'] ?? null,
        'sender' => $sender
    ]);
}


public function sendHistory()
{
    $data = $this->request->getJSON(true);

    $sender = $data['sender'] ?? '';
    $messages = $data['messages'] ?? [];

    if (empty($sender) || empty($messages)) {
        return $this->response->setJSON(['status' => 'error', 'message' => 'Datos incompletos']);
    }

    // 1. Armar el cuerpo del correo
    $body = "📞 Historial de conversación con: $sender\n\n";
    foreach ($messages as $msg) {
        $linea = "[{$msg['time']}] {$msg['sender']}:\n{$msg['message']}\n\n";
        $body .= $linea;
    }

  $email = \Config\Services::email();
$email->setFrom('preubasfondecom@gmail.com', 'Soporte Solibot');
$email->setTo('alexandra272gomez@gmail.com');
$email->setSubject("📩 Historial de conversación con $sender");
$email->setMessage(nl2br(htmlspecialchars($body)));

if (!$email->send()) {
    return $this->response->setJSON([
        'status' => 'error',
        'message' => 'Error al enviar correo: ' . $email->printDebugger(['headers'])
    ]);
}

    // 3. Enviar mensaje final de "Chat finalizado"
    $file = WRITEPATH . 'conversations.json';
    $conversations = file_exists($file) ? json_decode(file_get_contents($file), true) : [];

    if (isset($conversations[$sender])) {
        // 3. Enviar mensaje final de "Chat finalizado" al cliente por WhatsApp
     $this->sendToWhatsApp($sender, '✅ Chat finalizado. ¡Gracias por comunicarte con nosotros!');


        // 4. Guardar el mensaje y luego eliminar la conversación
       file_put_contents($file, json_encode($conversations, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

        // Borrar la conversación completa
        unset($conversations[$sender]);
       file_put_contents($file, json_encode($conversations, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
       
    }

    return $this->response->setJSON(['status' => 'success']);
}

   private function checkUserInactivity()
{
    $file = WRITEPATH . 'conversations.json';
    $logFile = WRITEPATH . 'processed_messages.log';
    $webhookLogFile = WRITEPATH . 'webhook_log.txt';

    $conversations = file_exists($file) ? json_decode(file_get_contents($file), true) : [];

    $currentTime = time();
    $inactivityLimit = 600; // 🔄 2 minutos para marcar como inactiva
    $deleteLimit = 600;     // 🧹 10 minutos para eliminar completamente

    foreach ($conversations as $from => $conversation) {
        if (isset($conversation['last_activity'])) {
            $lastActivity = $conversation['last_activity'];

            if ($lastActivity < $currentTime - $deleteLimit) {
                unset($conversations[$from]);

                // Limpiar processed_messages.log
                if (file_exists($logFile)) {
                    $lines = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                    $filtered = array_filter($lines, fn($line) => strpos($line, $from) === false);
                    file_put_contents($logFile, implode(PHP_EOL, $filtered) . PHP_EOL);
                }

                // Limpiar webhook_log.txt
                if (file_exists($webhookLogFile)) {
                    $lines = file($webhookLogFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                    $filtered = array_filter($lines, fn($line) => strpos($line, $from) === false);
                    file_put_contents($webhookLogFile, implode(PHP_EOL, $filtered) . PHP_EOL);
                }

                // Eliminar archivo de mensajes individuales
                $msgFile = WRITEPATH . 'messages_' . $from . '.json';
                if (file_exists($msgFile)) {
                    unlink($msgFile);
                }

            } elseif ($lastActivity < $currentTime - $inactivityLimit) {
                $conversations[$from]['status'] = 'inactive';
            }
        }
    }

    file_put_contents($file, json_encode($conversations, JSON_PRETTY_PRINT));
}


    // Marcar conversación para asignación de asesor
    private function markForAdvisorAssignment($currentFlow)
    {
        // Aquí puedes implementar lógica adicional para notificar a administradores
        // sobre conversaciones que requieren asignación de asesor
        file_put_contents(WRITEPATH . 'advisor_requests.txt', 
            "Nueva solicitud de asesor: " . date('Y-m-d H:i:s') . "\n", FILE_APPEND);
    }


    public function getAdvisors()
{

    $advisors = $this->userModel
        ->where('role', 'advisor')
        ->select('id, name') // puedes cambiar 'name' por el campo correcto
        ->findAll();

    return $this->response->setJSON([
        'status' => 'success',
        'advisors' => $advisors
    ]);
}

public function assign()
{
    $data = $this->request->getJSON(true);
    $sender = $data['sender'] ?? '';
    $userId = $data['user_id'] ?? 0;
    $priority = $data['priority'] ?? false;
    $observation = trim($data['observation'] ?? '');

    if (!$sender || !$userId) {
        return $this->response->setJSON(['status' => 'error', 'message' => 'Datos inválidos']);
    }

    // Validar asesor
    $user = $this->userModel->find($userId);
    if (!$user || $user['role'] !== 'advisor') {
        return $this->response->setJSON(['status' => 'error', 'message' => 'El usuario no es asesor válido']);
    }

   /* // Verificar si ya está asignado
    $existingAssignment = $this->db->table('chat_assignments')
        ->where('sender', $sender)
        ->get()
        ->getRow();

    if ($existingAssignment) {
        return $this->response->setJSON([
            'status' => 'error',
            'message' => 'Esta conversación ya fue asignada a otro asesor.'
        ]);
    }

    // Insertar nueva asignación
    $this->db->table('chat_assignments')->insert([
        'sender'      => $sender,
        'user_id'     => $userId,
        'assigned_at' => date('Y-m-d H:i:s')
    ]);*/

    // 🔽 Crear el ticket basado en la conversación
    $conversationsFile = WRITEPATH . 'conversations.json';
    $ticketsFile = WRITEPATH . 'tickets.json';

    $conversations = file_exists($conversationsFile) ? json_decode(file_get_contents($conversationsFile), true) : [];
    $tickets = file_exists($ticketsFile) ? json_decode(file_get_contents($ticketsFile), true) : [];

    if (!isset($conversations[$sender])) {
        return $this->response->setJSON(['status' => 'error', 'message' => 'No se encontró la conversación asociada']);
    }

    $conv = $conversations[$sender];
    $messages = $conv['messages'] ?? [];
    $lastMessage = end($messages);

    $ticket = [
        'id' => uniqid(),
        'ticket_number' => $this->generateTicketNumber(),
        'platform' => $conv['platform'] ?? 'desconocido',
        'user_platform_id' => $conv['user_platform_id'] ?? '',
        'user_name' => $conv['user_name'] ?? 'Usuario',
        'user_phone' => $sender,
        'subject' => 'Asignación manual desde conversación',
        'description' => $observation ?: ($lastMessage['message'] ?? 'Sin descripción'),
        'status' => 'open',
        'priority' => $priority ? 'Alta' : 'Media',
        'assigned_to' => $userId,
        'category' => null,
        'created_at' => date('Y-m-d H:i:s'),
        'updated_at' => date('Y-m-d H:i:s'),
        'assigned_at' => date('Y-m-d H:i:s')
    ];

    $tickets[] = $ticket;
    file_put_contents($ticketsFile, json_encode($tickets, JSON_PRETTY_PRINT));

    return $this->response->setJSON(['status' => 'success']);
}
 private function generateTicketNumber()
    {
        return 'TK-' . date('Ymd') . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
    }




}
//