<?php

namespace App\Models;

use CodeIgniter\Model;

class AdvisorAssignmentModel extends Model
{
    protected $table = 'advisor_assignments';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    protected $allowedFields = [
        'flow_id', 'flow_key', 'user_id'
    ];

    protected bool $allowEmptyInserts = false;
    protected bool $updateOnlyChanged = true;

    protected array $casts = [];
    protected array $castHandlers = [];

    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';

    protected $validationRules = [
        'flow_id' => 'required|integer',
        'flow_key' => 'required|max_length[100]',
        'user_id' => 'required|integer',
    ];

    protected $validationMessages = [];
    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    protected $allowCallbacks = true;
    protected $beforeInsert = [];
    protected $afterInsert = [];
    protected $beforeUpdate = [];
    protected $afterUpdate = [];
    protected $beforeFind = [];
    protected $afterFind = [];
    protected $beforeDelete = [];
    protected $afterDelete = [];

    /**
     * Obtener asesores asignados a un flujo específico
     */
    public function getAdvisorsByFlow($flowId,$firstNewFlow)
    {
        return $this->select('advisor_assignments.*, users.username, users.name')
                    ->join('users', 'users.id = advisor_assignments.user_id')
                    ->where('advisor_assignments.flow_id', $flowId)
                    ->where('advisor_assignments.flow_key', $firstNewFlow)
                    ->findAll();
    }

    /**
     * Obtener flujos asignados a un asesor específico
     */
    public function getFlowsByAdvisor($userId)
    {
        return $this->select('advisor_assignments.*, flows.name as flow_name, flows.flow_key')
                    ->join('flows', 'flows.id = advisor_assignments.flow_id')
                    ->where('advisor_assignments.user_id', $userId)
                    ->where('flows.is_active', true)
                    ->findAll();
    }

    /**
     * Asignar múltiples asesores a un flujo
     */
    public function assignAdvisorsToFlow($flowId, $flowKey, $advisorIds)
    {
        // Primero eliminar asignaciones existentes
        $this->where('flow_id', $flowId)->delete();

        // Insertar nuevas asignaciones
        $assignments = [];
        foreach ($advisorIds as $advisorId) {
            $assignments[] = [
                'flow_id' => $flowId,
                'flow_key' => $flowKey,
                'user_id' => $advisorId
            ];
        }

        if (!empty($assignments)) {
            return $this->insertBatch($assignments);
        }

        return true;
    }

    /**
     * Obtener un asesor aleatorio asignado a un flujo
     */
    public function getRandomAdvisorForFlow($flowKey)
    {
        $advisors = $this->select('users.*')
                         ->join('users', 'users.id = advisor_assignments.user_id')
                         ->join('flows', 'flows.id = advisor_assignments.flow_id')
                         ->where('flows.flow_key', $flowKey)
                         ->where('flows.is_active', true)
                         ->where('users.is_active', true)
                         ->findAll();

        if (!empty($advisors)) {
            return $advisors[array_rand($advisors)];
        }

        return null;
    }

    /**
     * Obtener todas las asignaciones agrupadas por flujo
     */
    public function getAllAssignmentsGrouped()
    {
        $assignments = $this->select('advisor_assignments.*, flows.name as flow_name, flows.flow_key, users.username, users.name')
                            ->join('flows', 'flows.id = advisor_assignments.flow_id')
                            ->join('users', 'users.id = advisor_assignments.user_id')
                            ->where('flows.is_active', true)
                            ->orderBy('flows.order_index', 'ASC')
                            ->findAll();

                             


        $grouped = [];
        foreach ($assignments as $assignment) {
            $flowKey = $assignment['flow_key'];
            if (!isset($grouped[$flowKey])) {
                $grouped[$flowKey] = [
                    'flow_id' => $assignment['flow_id'],
                    'flow_name' => $assignment['flow_name'],
                    'flow_key' => $assignment['flow_key'],
                    'advisors' => []
                ];
            }
            $grouped[$flowKey]['advisors'][] = [
                'user_id' => $assignment['user_id'],
                'username' => $assignment['username'],
                'full_name' => $assignment['name']
            ];
        }

        return $grouped;
    }
}