<?php

namespace App\Models;

use CodeIgniter\Model;

class FlowModel extends Model
{
    protected $table = 'flows';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    protected $allowedFields = [
        'flow_key', 'name', 'type', 'description', 'message', 'options',
        'redirect_url', 'parent_id', 'order_index', 'is_active', 'requires_advisor'
    ];

    protected bool $allowEmptyInserts = false;
    protected bool $updateOnlyChanged = true;

    protected array $casts = [
        'options' => 'json',
        'is_active' => 'boolean',
        'requires_advisor' => 'boolean',
    ];
    protected array $castHandlers = [];

    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    protected $deletedField = 'deleted_at';

    protected $validationRules = [
        'flow_key' => 'required|max_length[100]|is_unique[flows.flow_key,id,{id}]',
        'name' => 'required|max_length[255]',
        'type' => 'required|in_list[menu,submenu,action,redirect]',
        'message' => 'required',
    ];

    protected $validationMessages = [
        'flow_key' => [
            'required' => 'La clave del flujo es obligatoria',
            'is_unique' => 'Esta clave de flujo ya existe'
        ],
        'name' => [
            'required' => 'El nombre del flujo es obligatorio'
        ],
        'message' => [
            'required' => 'El mensaje del chatbot es obligatorio'
        ]
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    protected $allowCallbacks = true;
    protected $beforeInsert = [];
    protected $afterInsert = [];
    protected $beforeUpdate = [];
    protected $afterUpdate = [];
    protected $beforeFind = [];
    protected $afterFind = [];
    protected $beforeDelete = [];
    protected $afterDelete = [];

    /**
     * Obtener flujos activos ordenados
     */
    public function getActiveFlows($parentId = null)
    {
        $builder = $this->where('is_active', true);
        
        if ($parentId !== null) {
            $builder->where('parent_id', $parentId);
        }
        
        return $builder->orderBy('order_index', 'ASC')->findAll();
    }

    /**
     * Obtener flujo por clave
     */
    public function getFlowByKey($flowKey)
    {
        return $this->where('flow_key', $flowKey)
                    ->where('is_active', true)
                    ->first();
    }

    /**
     * Obtener flujos hijos
     */
    public function getChildFlows($parentId)
    {
        return $this->where('parent_id', $parentId)
                    ->where('is_active', true)
                    ->orderBy('order_index', 'ASC')
                    ->findAll();
    }

    /**
     * Obtener estadísticas de flujos
     */
    public function getFlowStats()
    {
        return [
            'total' => $this->countAllResults(false),
            'active' => $this->where('is_active', true)->countAllResults(false),
            'menus' => $this->where('type', 'menu')->countAllResults(false),
            'submenus' => $this->where('type', 'submenu')->countAllResults(false),
            'actions' => $this->where('type', 'action')->countAllResults(false),
            'redirects' => $this->where('type', 'redirect')->countAllResults(false),
        ];
    }

    /**
     * Formatear flujo para WhatsApp
     */
 public function formatFlowForWhatsApp($flow)
{
    if (!$flow || !isset($flow['message'])) {
        return null;
    }

    // Iniciar con el mensaje base
    $formattedMessage = $flow['message'];

    // Procesar las opciones del flujo si existen
    if (!empty($flow['options']) && is_array($flow['options'])) {
        foreach ($flow['options'] as $option) {
            if (is_object($option)) {
                // En tu caso las opciones son stdClass, por eso usamos ->
                $formattedMessage .= "\n" . $option->key . ". " . $option->text;
            } elseif (is_array($option)) {
                $formattedMessage .= "\n" . $option['key'] . ". " . $option['text'];
            }
        }
    }

    return $formattedMessage;
}



    /**
     * Crear flujos predeterminados
     */
    public function createDefaultFlows()
    {
        $defaultFlows = [
            [
                'flow_key' => 'main_menu',
                'name' => 'Menú Principal',
                'type' => 'menu',
                'description' => 'Menú principal del chatbot con todas las opciones de servicio',
                'message' => "🏦 *¡Bienvenido a Chat Solibot!*\n\nSoy tu asistente virtual. Selecciona el servicio que necesitas:",
                'options' => [
                    ['key' => '1', 'text' => '💳 Crédito', 'action' => 'submenu', 'value' => 'credito_menu'],
                    ['key' => '2', 'text' => '💰 Cartera', 'action' => 'submenu', 'value' => 'cartera_menu'],
                    ['key' => '3', 'text' => '🏦 Ahorros y CDAT', 'action' => 'submenu', 'value' => 'ahorros_menu'],
                    ['key' => '4', 'text' => '🤝 Auxilios', 'action' => 'submenu', 'value' => 'auxilios_menu'],
                    ['key' => '5', 'text' => '📋 Otros servicios', 'action' => 'submenu', 'value' => 'otros_menu'],
                    ['key' => '0', 'text' => '👨‍💼 Hablar con un asesor', 'action' => 'advisor', 'value' => '']
                ],
                'parent_id' => null,
                'order_index' => 1,
                'is_active' => true,
                'requires_advisor' => false
            ],
            [
                'flow_key' => 'credito_menu',
                'name' => 'Menú de Crédito',
                'type' => 'submenu',
                'description' => 'Opciones relacionadas con servicios de crédito',
                'message' => "💳 *SERVICIOS DE CRÉDITO*\n\nSelecciona la opción que necesitas:",
                'options' => [
                    ['key' => '1', 'text' => '📄 Descargar reglamento de crédito', 'action' => 'url', 'value' => 'https://ejemplo.com/reglamento-credito.pdf'],
                    ['key' => '2', 'text' => '📝 Radicar solicitud de crédito', 'action' => 'action', 'value' => 'solicitud_credito'],
                    ['key' => '3', 'text' => '👨‍💼 Comunicar con un asesor', 'action' => 'advisor', 'value' => ''],
                    ['key' => '0', 'text' => '⬅️ Volver al menú principal', 'action' => 'submenu', 'value' => 'main_menu']
                ],
                'parent_id' => null, // Se actualizará después de crear el menú principal
                'order_index' => 2,
                'is_active' => true,
                'requires_advisor' => false
            ],
            [
                'flow_key' => 'cartera_menu',
                'name' => 'Menú de Cartera',
                'type' => 'submenu',
                'description' => 'Opciones para consultas de cartera y estados de cuenta',
                'message' => "💰 *SERVICIOS DE CARTERA*\n\nSelecciona la opción que necesitas:",
                'options' => [
                    ['key' => '1', 'text' => '📄 Descargar estado de cuenta', 'action' => 'url', 'value' => 'https://ejemplo.com/estado-cuenta'],
                    ['key' => '2', 'text' => '💵 Consultar saldo actual', 'action' => 'action', 'value' => 'consultar_saldo'],
                    ['key' => '3', 'text' => '👨‍💼 Comunicar con un asesor', 'action' => 'advisor', 'value' => ''],
                    ['key' => '0', 'text' => '⬅️ Volver al menú principal', 'action' => 'submenu', 'value' => 'main_menu']
                ],
                'parent_id' => null,
                'order_index' => 3,
                'is_active' => true,
                'requires_advisor' => false
            ],
            [
                'flow_key' => 'ahorros_menu',
                'name' => 'Menú de Ahorros y CDAT',
                'type' => 'submenu',
                'description' => 'Servicios de ahorros y certificados de depósito',
                'message' => "🏦 *AHORROS Y CDAT*\n\nSelecciona la opción que necesitas:",
                'options' => [
                    ['key' => '1', 'text' => '📋 Consultar reglamento de ahorro', 'action' => 'url', 'value' => 'https://ejemplo.com/reglamento-ahorro.pdf'],
                    ['key' => '2', 'text' => '📄 Descargar estado de cuenta de ahorros', 'action' => 'url', 'value' => 'https://ejemplo.com/estado-ahorros'],
                    ['key' => '3', 'text' => '💎 Información sobre CDAT', 'action' => 'action', 'value' => 'info_cdat'],
                    ['key' => '4', 'text' => '👨‍💼 Comunicar con un asesor', 'action' => 'advisor', 'value' => ''],
                    ['key' => '0', 'text' => '⬅️ Volver al menú principal', 'action' => 'submenu', 'value' => 'main_menu']
                ],
                'parent_id' => null,
                'order_index' => 4,
                'is_active' => true,
                'requires_advisor' => false
            ],
            [
                'flow_key' => 'auxilios_menu',
                'name' => 'Menú de Auxilios',
                'type' => 'submenu',
                'description' => 'Servicios de auxilios y beneficios',
                'message' => "🤝 *SERVICIOS DE AUXILIOS*\n\nSelecciona la opción que necesitas:",
                'options' => [
                    ['key' => '1', 'text' => '📄 Descargar reglamento de auxilios', 'action' => 'url', 'value' => 'https://ejemplo.com/reglamento-auxilios.pdf'],
                    ['key' => '2', 'text' => '🔍 Consultar estado de solicitudes', 'action' => 'action', 'value' => 'estado_auxilios'],
                    ['key' => '3', 'text' => '👨‍💼 Comunicar con un asesor', 'action' => 'advisor', 'value' => ''],
                    ['key' => '0', 'text' => '⬅️ Volver al menú principal', 'action' => 'submenu', 'value' => 'main_menu']
                ],
                'parent_id' => null,
                'order_index' => 5,
                'is_active' => true,
                'requires_advisor' => false
            ],
            [
                'flow_key' => 'otros_menu',
                'name' => 'Otros Servicios',
                'type' => 'submenu',
                'description' => 'Otros servicios y consultas generales',
                'message' => "📋 *OTROS SERVICIOS*\n\nPor favor, describe brevemente tu solicitud o selecciona una opción:",
                'options' => [
                    ['key' => '1', 'text' => '✍️ Escribir solicitud específica', 'action' => 'action', 'value' => 'solicitud_libre'],
                    ['key' => '2', 'text' => 'ℹ️ Información general', 'action' => 'action', 'value' => 'info_general'],
                    ['key' => '3', 'text' => '👨‍💼 Comunicar con un asesor', 'action' => 'advisor', 'value' => ''],
                    ['key' => '0', 'text' => '⬅️ Volver al menú principal', 'action' => 'submenu', 'value' => 'main_menu']
                ],
                'parent_id' => null,
                'order_index' => 6,
                'is_active' => true,
                'requires_advisor' => false
            ]
        ];

        // Insertar flujos
        foreach ($defaultFlows as $flow) {
            $this->insert($flow);
        }

        return true;
    }

    /**
     * Verificar si existen flujos predeterminados
     */
    public function hasDefaultFlows()
    {
        return $this->where('flow_key', 'main_menu')->first() !== null;
    }
}